package orchestrator;

import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import java.util.Vector;

import service.ComposedTransition;
import service.ComposedState;
import service.SimpleTransition;
import service.SimpleState;
import service.State;
import service.TransitionSystem;
import simulator.Simulation;
import simulator.SimulationFactoryImpl;

public class OrchestratorStandardImpl extends Orchestrator {
	
	
	/**
	 * Class to sort states according to this order:
	 * first they are ordered according to the lexicographical order of the name of the first state
	 * if the names of the first state are equals then the lexicographical order of the name of the second state is evaluated
	 * @author daniele
	 *
	 */
	private class SimulationTuplesComparator implements Comparator<State[]>{

		/* (non-Javadoc)
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		@Override
		public int compare(State[] o1, State[] o2) {
			if(o1[0].getName().compareTo(o2[0].getName())<0)
				return -1;
			else if(o1[0].getName().compareTo(o2[0].getName())==0){
				return o1[1].getName().compareTo(o2[1].getName());
			}else
				return 1;
		}
		
	}
	
	/**
	 * Class to sort array of strings according to this order:
	 * the lexicographical order of the first elements is evaluated at first.
	 * If the two elements above are equals
	 * the lexicographical order of the second elements is evaluated.
	 * If also these are equals
	 * the lexicographical order of the third elements is evaluated.
	 * At last if also the third elements are equals
	 * the lexicographical odrder of the fourth elements is evaluated.
	 * @author daniele
	 *
	 */
	private class CompositionComparator implements Comparator<String[]>{

		/* (non-Javadoc)
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		@Override
		public int compare(String[] arg0, String[] arg1) {
			if(arg0[0].compareTo(arg1[0])<0)
				return -1;
			else if(arg0[0].compareTo(arg1[0])==0){
				if(arg0[1].compareTo(arg1[1])<0)
					return -1;
				else if(arg0[1].compareTo(arg1[1])==0){
					if(arg0[2].compareTo(arg1[2])<0)
						return -1;
					else if(arg0[2].compareTo(arg1[2])==0){
						return arg0[3].compareTo(arg1[3]);
					}
					else
						return 1;
				}
				else
					return 1;
			}
			else return 1;
				
		}
		
	}
	
	Simulation sym;
	SimpleState currentTargetState;
	ComposedState currentServicesState;
	ComposedTransition lastExecutedAction=null;
	TransitionSystem<SimpleState, SimpleTransition>[] available;
	/**
	 * Creates a new orchestrator for the given target services and available services
	 * @param target the transition system of the target service
	 * @param availableServices the transition systems of the available services
	 * @throws Exception an orchestrator cannot be created with the given parameters
	 */
	public OrchestratorStandardImpl(TransitionSystem<SimpleState, SimpleTransition> target,
			TransitionSystem<SimpleState, SimpleTransition>[] availableServices) throws Exception {
		super(target, availableServices);
		this.available=availableServices;
		this.sym=(new SimulationFactoryImpl()).createSimulationStandard(target,availableServices);
		this.currentTargetState=target.getInitial();
		this.currentServicesState=this.sym.getCompositionTs().getInitial();
	}

	@Override
	public void doAction(SimpleTransition action) throws Exception{
		Set<SimpleTransition> actions=this.sym.getTargetTs().getActionsOf(currentTargetState);
		if(!actions.contains(action))
			throw new Exception("Selected action is not available");
		Iterator<ComposedTransition> it=this.sym.getCompositionTs().getActionsOf(currentServicesState).iterator();
		ComposedTransition current=null;
		while(it.hasNext()){
			current=it.next();
			if(current.actionSimulates(action) && sym.getSimilarStates(action.getStateTo()).contains(current.getStateTo())){
				this.currentServicesState=current.getStateTo();
				this.currentTargetState=(SimpleState)action.getStateTo();
				this.lastExecutedAction=current;
				break;
			}
		}
	}

	public SimpleState getCurrentTargetState() {
		return currentTargetState;
	}

	public ComposedState getCurrentServicesState() {
		return currentServicesState;
	}
	public ComposedTransition getLastExecutedAction() {
		return lastExecutedAction;
	}

	public TransitionSystem<ComposedState, ComposedTransition> getCompositionTs() {
		
		return sym.getCompositionTs();
	}
	public Set<State[]> getAllSimilarState(){
		return sym.getAllSimilarState();
	}

	/* (non-Javadoc)
	 * @see orchestrator.Orchestrator#getSimulationDataToExport()
	 */
	@Override
	public Vector<String[]> getSimulationDataToExport() {
		Iterator<State[]> it=sym.getAllSimilarState().iterator();
		LinkedList<State[]> sorter=new LinkedList<State[]>();
		//String composedStateString;
		//copies the content of the relation to a list
		while(it.hasNext()){
			sorter.add(it.next());
		}
		//sorts it
		Collections.sort(sorter, new SimulationTuplesComparator());
		
		it=sorter.iterator();
		
		//puts the content of the sorted list in a vector
		Vector<String[]> result=new Vector<String[]>();
		State[] current=null;
		while(it.hasNext()){
			current=it.next();
			result.add(new String[]{current[0].getName(),current[1].getName()});
			//TODO UNCOMMENT IF NECESSARY and delete the row above (only ONE!!!!!!)
			//AND UNCOMMENT DECLARATION OF composedStateString		
			/*try {

				composedStateString = "";
				for (int i = 0; i < available.length; i++) {
					if (i > 0)
						composedStateString += ",";
					composedStateString += ((ComposedState) current[1]).getServicesState(i).getName();
				}
				result.add(new String[] { current[0].getName(),
						composedStateString });
			} catch (Exception e) {
			}*/
		}
		
		
		return result;
	}

	/* (non-Javadoc)
	 * @see orchestrator.Orchestrator#getCompositionDataToExport()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Vector<String[]> getCompositionDataToExport() {
		Vector<String[]> result=new Vector<String[]>();
		Iterator<SimpleState> itOverTSstates=sym.getTargetTs().getAllStates().iterator();
		Iterator<ComposedState> itOverSimilarStates;
		Iterator<SimpleTransition> itOverTSStateActions;
		Iterator<ComposedTransition> itOverAvailableActions;
		Iterator<String> itOverStrings;
		String firstArg, secondArg, thirdArg, fourthArg;//string to compose array of string
		LinkedList<String[]> sorter=new LinkedList<String[]>();
		LinkedList<String> serviceList=new LinkedList<String>();
		SimpleState currentTS=null;
		ComposedState currentComposedState=null;
		SimpleTransition currentAction=null;
		ComposedTransition currentComposedAction=null;
		try{
			while(itOverTSstates.hasNext()){
				//select the target service state
				currentTS=itOverTSstates.next();

				firstArg=currentTS.getName();
				itOverSimilarStates=sym.getSimilarStates(currentTS).iterator();
				while(itOverSimilarStates.hasNext()){
					currentComposedState=itOverSimilarStates.next();
					secondArg="";
					
					State states[]=currentComposedState.getStates();
					for(int i=0; i<states.length; i++){
						if(i>0)
							secondArg+=",";
						secondArg+=states[i].getName();
					}

					itOverTSStateActions=sym.getTargetTs().getActionsOf(currentTS).iterator();
					while(itOverTSStateActions.hasNext()){
						currentAction=itOverTSStateActions.next();	
						thirdArg=currentAction.getName();
						fourthArg="";
						serviceList=new LinkedList<String>();
						LinkedList<String> serviceListToRemove=new LinkedList<String>();
						boolean firstAdd=true;
						
						itOverAvailableActions=sym.getCompositionTs().getActionsOf(currentComposedState).iterator();
						//giovanni's code
						while(itOverAvailableActions.hasNext()){

							currentComposedAction=itOverAvailableActions.next();
							int index=getServiceIndex((TransitionSystem<SimpleState, SimpleTransition>) currentComposedAction.getService());
							if(currentAction.actionSimulates(currentComposedAction) && sym.getSimilarStates(currentAction.getStateTo()).contains(currentComposedAction.getStateTo())){
								if(!serviceList.contains(available[index].getName()))
									serviceList.add(available[index].getName());
							}else if(currentAction.actionSimulates(currentComposedAction)){
								serviceListToRemove.add(available[index].getName());
							}
						}
						//
						/* daniele's code
						while(itOverAvailableActions.hasNext()){

							currentComposedAction=itOverAvailableActions.next();
							int index=getServiceIndex((TransitionSystem<SimpleState, SimpleAction>) currentComposedAction.getService());
							if(currentAction.actionSimulates(currentComposedAction) && sym.getSimilarStates(currentAction.getStateTo()).contains(currentComposedAction.getStateTo())){
								if(!serviceList.contains(available[index].getName()))
									serviceList.add(available[index].getName());
							}	
						}*/
							serviceList.removeAll(serviceListToRemove);
							Collections.sort(serviceList);
							
							itOverStrings=serviceList.iterator();
							while(itOverStrings.hasNext()){
								if(!firstAdd){
									fourthArg+=",";
								}else{
									firstAdd=false;
								}
								fourthArg+=itOverStrings.next();
							}
							
							sorter.add(new String[]{firstArg, secondArg, thirdArg, fourthArg});
					}
				}
			}
			
			Collections.sort(sorter, new CompositionComparator());
			
			Iterator<String[]> it=sorter.iterator();
			while(it.hasNext()){
				result.add(it.next());
			}
			
		}catch(Exception e){}
		return result;
	}
	
	/**
	 * Returns the index of the requested service
	 * @param service 
	 * @return an index that represents its position in the array, -1 if the service is not present
	 */
	private int getServiceIndex(TransitionSystem<SimpleState, SimpleTransition> service){
		for(int i=0; i<this.available.length; i++){
			if(available[i]==service)
				return i;
		}
		return -1;
	}

	/* (non-Javadoc)
	 * @see orchestrator.Orchestrator#getAsyncProduct()
	 */
	@Override
	public TransitionSystem<ComposedState, ComposedTransition> getAsyncProduct() {
		return sym.getAsyncProduct();
	}
}
